:- module(pendulum,_,[fuzzy,assertions,isomodes,regtypes]).

:- use_module(library(aggregates)). 
:- use_module(library(lists)). 

:- comment(doinclude,angle_zero/2).
:- comment(doinclude,angle_pn/2).
:- comment(doinclude,speed_zero/2).
:- comment(doinclude,speed_pn/2).
:- comment(doinclude,fam/4).
:- comment(doinclude,list_all/3).
:- comment(doinclude,centroid/2).

:- regtype region(R) 
#"@var{R} describes the possible region where the inverted pendulum is moving.
".
region(neg).
region(zero).
region(pos).


:- pred angle_zero(-Angle,+FuzzyValue)::float*float
#"
Compact representation for the fuzzy function relating @var{Angle} and
 @var{FuzzyValue} for the Zero class [Figure 9.9a]
".
angle_zero :# fuzzy_predicate( [(-2,0),(0,1),(2,0)]).


:- pred angle_pn(-Angle,+FuzzyValue)::float*float
#"
Compact representation for the fuzzy function relating @var{Angle} and
 @var{FuzzyValue} for the Positive and Negative classes [Figure 9.9a]
".
angle_pn :# fnot angle_zero/2. 


:- pred speed_zero(-Speed,+FuzzyValue)::float*float
#"
Compact representation for the fuzzy function relating @var{Speed} and
 @var{FuzzyValue} for the Zero class [Figure 9.9b]
".
speed_zero :# fuzzy_predicate( [(-5,0),(0,1),(5,0)]).


:- pred speed_pn(-Speed,+FuzzyValue)::float*float
#"
Compact representation for the fuzzy function relating @var{Speed} and
 @var{FuzzyValue} for the P/N classes, described as the negation of the 
previous predicate [Figure 9.9b]
".
speed_pn :# fnot speed_zero/2.


:- pred fam(-Angle,-Speed,Movement,FuzzyValue)::int*int*region*float
#"For the given @var{Angle} and @var{Speed} we calculate a @var{FuzzyValue} 
corresponding in either the zero, positive or negative categories represented 
by @var{Movement}
".
fam(Angle,Speed,neg,FV) :-
	Angle .<.0,
	angle_pn(Angle,FuzzyAngle),
	speed_zero(Speed,FuzzySpeed),
	min(FuzzyAngle,FuzzySpeed,FV).

fam(Angle,Speed,neg,FV) :-
	Speed .<.0,
	speed_pn(Speed,FuzzySpeed),
	angle_zero(Angle,FuzzyAngle),
	min(FuzzyAngle,FuzzySpeed,FV).


fam(Angle,Speed,zero,FV) :-
	angle_zero(Angle,FuzzyAngle),
	speed_zero(Speed,FuzzySpeed),
	min(FuzzyAngle,FuzzySpeed,FV).

fam(Angle,Speed,zero,FV) :-
	Angle .>.0,
	Speed .<.0,
	angle_pn(Angle,FuzzyAngle),
	speed_pn(Speed,FuzzySpeed),
	min(FuzzyAngle,FuzzySpeed,FV).

fam(Angle,Speed,zero,FV) :-
	Angle .<.0,
	Speed .>.0,
	angle_pn(Angle,FuzzyAngle),
	speed_pn(Speed,FuzzySpeed),
	min(FuzzyAngle,FuzzySpeed,FV).

fam(Angle,Speed,big_pos,FV) :-
	Angle .>.0,
	Speed .>. 0,
	angle_pn(Angle,FuzzyAngle),
	speed_pn(Speed,FuzzySpeed),
	min(FuzzyAngle,FuzzySpeed,FV).

fam(Angle,Speed,big_neg,FV) :-
	Angle .<.0,
	Speed .<. 0,
	fam(-Angle,-Speed,big_pos,FV).

fam(Angle,Speed,pos,FV) :-
	Angle .>.0,
	angle_pn(Angle,FuzzyAngle),
	speed_zero(Speed,FuzzySpeed),
	min(FuzzyAngle,FuzzySpeed,FV).

fam(Angle,Speed,pos,FV) :-
	Speed .>.0,
	speed_pn(Speed,FuzzySpeed),
	angle_zero(Angle,FuzzyAngle),
	min(FuzzyAngle,FuzzySpeed,FV).


:- pred list_all(Angle,Speed,Fuzzified)::int*int*list((region,float))
#"
Finds all possible fuzzified values for certain @var{Angle} and @var{Speed} by
 searching in the three categories. 
".
list_all(Angle,Speed,Result) :-
	findall((Movement,FV),fam(Angle,Speed,Movement,FV),Result).
	

prune([(X,FV1)|Rest],Rest):-
	nth(_,Rest,(X,FV2)),
	FV1 .<. FV2,
	!.
prune([H|Rest],[H|Result]):-
	prune(Rest,Result).


:- pred centroid(FuzzyThreesome, +Result)::list(region*float)*float
#"
Apply the centroid defuzzification technique to the union of the areas formed
 by the (maximum) fuzzy values in each of the categories, stored in
 @var{FuzzyThreesome}. Since we are only interested in the x-coordinate of the
 result it can be demonstrated that there are seven possible types of @var{Result}
".
centroid([(neg,A),(zero,_B),(pos,C)],Res):-
	A .=. C,
	Res .=. 0,!.

centroid([(neg,A),(zero,_B),(pos,C)],Res):-
	A .=. B,
	C .<. B,
	Res .=. -2,!.

centroid([(neg,A),(zero,B),(pos,C)],Res):-
	A .>. B,
	B .=. C,
	Res .=. -4,!.

centroid([(neg,A),(zero,B),(pos,C)],Res):-
	A .>. B,
	B .>. C,
	Res .=. -5,!.

centroid([(neg,A),(zero,B),(pos,C)],Res):-
	centroid([(neg,C),(zero,B),(pos,A)],NRes),
	Res .=. -NRes.


:- pred crisp_output(-Angle, -Speed, +Output)::float*float*(float*region)
#"
Ultimate goal of the whole process: given @var{Angle} and @var{Speed}, return 
the recommended movement described in @var{Output}.
".
crisp_output(Angle,Speed,Output):-
	list_all(Angle,Speed,List),
	prune(List,N_Z_P),
	centroid(N_Z_P,Output).
