:-module(search,
	[
	 path/2,
	 type_search/1,
	 get_example/1
	],
	[assertions,isomodes,regtypes]).

% abstract data types
:-use_module('./adt/queue.pl').
:-use_module('./adt/set.pl').
:-use_module('./adt/bag.pl').

%%% Games

% :-use_module('./games/n_puzzle.pl').
:-use_module('./games/cannibals.pl').

% Preprocessing stuff
:-use_module(library(hiordlib)).
:-use_module(library(write)).
:-discontiguous [search/4].
:-set_prolog_flag(multi_arity_warnings,off).

:- comment(title, "Graph search").
:- comment(author,"Mario Mendez").

:- comment(doinclude,search/4).
:- comment(doinclude,update_open/5).
:- comment(doinclude,update_closed/6).
:- comment(doinclude,to_type_bag/2).


:- comment(module,"This module implements the three types of search
methods: @bf{depth-first search}, @bf{breadth-first search} 
and @bf{best-first search}.").


:- pred path(-Node,-Type):: term * type_search 
#"Outputs a path from a start node @var{Node} to a goal node, if
  such solution exists. The search method used depends on @var{Type}. If
  @var{Type} is @var{'depth'} the algorithm used is @concept{depth-first
  search}. If @var{Type} is @var{'breadth'} the algorithm is
  @concept{breadth-first search}. Finally, when @var{Type} is @var{'best'}
  the algorithm is @concept{best-first search}.
  @item @em{Code:}
  @includedef{path/2}.

".

path(Node,Type) :-
	to_type_bag(Type,T),
	empty(Open,T),
	insert(Node,Open,NOpen,T),
	empty_set(Closed),
        q_empty(Path),
	search(NOpen,Closed,Path,Type).



:- pred search(-Open,-Closed,+Path,-Type):: bag * set * queue * type_search
#"When @var{Type} is @bf{'depth'}, this predicate performs a
 @concept{depth-first search}, exploring all descendants
 of a node before any of its siblings. @var{Open} lists states that have
 been generated but whose children have not been examined yet. In this case,
 @var{Open} is a @bf{stack}. @var{Closed} records states that have already
 been examined to prevent loops. @var{Path} is the path from start
 node to a goal node.".

:- pred search(-Open,-Closed,+Path,-Type):: bag * set * queue * type_search
#"When @var{Type} is @bf{'breadth'}, this predicate is equivalent to a 
 @concept{breadth-first search}. This class of search explores the space in
 a layered way. Only when there are no more states to be
 explored at the actual level the algorithm moves to the next one. 
 @var{Open} lists states that have been generated but whose children
 have not been examined. In this case @var{Open} is a queue. @var{Closed}
 records states that have already been examined to prevent
 loops. @var{Path} is the path from start node to a goal node.  ".

:- pred search(-Open,-Closed,+Path,-Type):: bag * set * queue * type_search
#"When @var{Type} is @bf{'best'}, this predicate executes
 @concept{best-first search}. This class of search explores the most
 promising state at each iteration. 
 @concept{Best-first search} applies a @index{heuristic}
 evaluation function that sorts the states on @var{Open}. In this case 
 @var{Open} is a priority queue.

 @item @em{Code:}
 @includedef{search/4}.

".

search(Open,_,[],Type):-
	to_type_bag(Type,T),
	empty(Open,T),
	write('No solution found'),nl,!.
	
search(Open,_Closed,Path,Type):- 
	to_type_bag(Type,T),
	head(Node,Open,_NOpen,T),
	is_goal(Node),
	q_enqueue(Node,Path,NPath),
	write('Solution Path is: '), nl,
        print_path(NPath),
	q_length(NPath,L),
	write('Number of explored nodes: '), write(L),nl.

search(Open,Closed,Path,Type):-
	to_type_bag(Type,T),
	head(Node,Open,NOpen,T),	
	%% Output%%%%%%%%%%%%%%%%%%%%
 	%write('Explored Node: '),nl,
 	%print_path([Node]), 
	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
	get_children(Node,Children),
	%% Output%%%%%%%%%%%%%%%%%%%%
        % write('Children Nodes: '),nl,
        % print_path(Children),
	%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        union_set(Closed,[Node],NClosed),
        q_enqueue(Node,Path,NPath),
	update_closed(Children,NClosed,NNOpen,NNClosed,NNNOpen,Type),
	update_open(Children,NOpen,NClosed,NNOpen,Type),	
	search(NNNOpen,NNClosed,NPath,Type).


:- comment(update_open(Children,Open,Closed,New_Open,Type),"Checks for all the elements @var{Node} in @var{Children} whether they are in @var{Open} and
@var{Closed}, producing @var{New_Open} 

   @item @em{Code:}
   @includedef{update_open/5}
   @includedef{update_open_/5}
").
	  
:- pred update_open(-Children,-Open,-Closed,+New_Open,-Type):: list(board) *
	bag * bag * bag * type_search
#"When @var{Type} is @bf{best} it checks for all @var{Node} of
 @var{Children} if they already are in @var{Open}. If
 @var{Node} is not in @var{Open} or @var{Closed} then it is added to
 @var{Open} producing @var{New_Open}. If @var{Node} is in @var{Open}
 and it was reached by a shorter path, only this (better) state is preserved
 in @var{New_Open}. Otherwise @var{Open} is equal to @var{New_Open}.".

:- pred update_open(-Children,-Open,-Closed,+New_Open,-Type):: list(board)
	*bag * bag * bag * type_search
 #"When @var{Type} is @bf{breadth} or @bf{depth} it checks for all @var{Node}
of @var{Children} that @var{Node} is not on @var{Open} or @var{Closed}, thus
adding @var{Node} to @var{Open} producing @var{New_Open}. Otherwise @var{Open}
is equal to @var{New_Open}.".

update_open([],Open,_,Open,_).
update_open([Child|Children],Open,Closed,NNOpen,Type):-
	update_open_(Child,Open,Closed,NOpen,Type),
	update_open(Children,NOpen,Closed,NNOpen,Type).

% BestFS, the new state already exists but h_new < h_old
update_open_(Child,Open,_Closed,NNOpen,best):-  
	select_node(Child,Open,Child1),
	get_function(Child,HF),
	get_function(Child1,HF1),
	HF < HF1,
	to_type_bag(best,T),
	delete(Child1,Open,NOpen,T),
	insert(Child,NOpen,NNOpen,T),!.
	
% DFS and BFS: the state is already in Open or Closed
% BestFS     : the new state is already in Open but h_new >= h_old
update_open_(Child,Open,_Closed,Open,_):-  
	select_node(Child,Open,_),!.

update_open_(Child,_Open,Closed,_Open,_):-  
	select_node(Child,Closed,_),!.

% All types:   the state is not in Open
update_open_(Child,Open,_Closed,NOpen,Type):-  
	to_type_bag(Type,T),
	insert(Child,Open,NOpen,T).


:- pred update_closed(-Children,-Closed,-Open,+New_Closed,+New_Open,-Type)::
list(board) * set * bag * set * bag * type_search
#"A just processed state is added to the @var{Closed} list producing @var{New_Closed} for avoiding infinite loops in the algorithm.

   @item @em{Code:}
   @includedef{update_closed/6}
   @includedef{update_closed_/6}
".

update_closed([],Closed,Open,Closed,Open,_).
update_closed([Child|Children],Closed,Open,NNClosed,NNOpen,T):-
	update_closed_(Child,Closed,Open,NClosed,NOpen,T),
	update_closed(Children,NClosed,NOpen,NNClosed,NNOpen,T).

update_closed_(Child,Closed,Open,NClosed,NOpen,best):-
	select_node(Child,Closed,Child1) ->
	get_function(Child,HF),
	get_function(Child1,HF1),
	HF < HF1,
	to_type_bag(best,T),
	delete(Child1,Closed,NClosed,T),
	insert(Child,Open,NOpen,T).
	 
update_closed_(_,Closed,Open,Closed,Open,_).



:- pred to_type_bag(-Search,+Data_Structure):: type_search * type_bag
#"@var{Data_Structure} is the (symbolic) type associated to @var{Search}.

  @item @em{Code:}
  @includedef{to_type_bag/2}
".

to_type_bag(best,pqueue).
to_type_bag(breadth,queue).
to_type_bag(depth,stack).



:- comment(type_search/1,"Classes of search methods. 

  @item @em{Code:}
  @includedef{type_search/1}

").

:- regtype type_search(T) #"@var{T} is a class of search methods that can be: @concept{depth},
@concept{breadth} or @concept{best}.
".

type_search(best).
type_search(depth). 
type_search(breadth). 



% >Usage examples

% ** 8-puzzle **

% Example 1
% get_example(board(3,
%  	[tile(pos(1,1),2),tile(pos(1,3),3),tile(pos(2,1),1),tile(pos(2,2),8),
%  	 tile(pos(2,3),4),tile(pos(3,1),7),tile(pos(3,2),6),tile(pos(3,3),5)],
%  	 tile(pos(1,2),empty),
%  	 0)).

% Example2
% get_example(board(4,
%  	[tile(pos(1,1),2),tile(pos(1,2),8),tile(pos(1,3),3),tile(pos(2,1),1),
%  	 tile(pos(2,2),6),tile(pos(2,3),4),tile(pos(3,1),7),tile(pos(3,3),5)],
%  	 tile(pos(3,2),empty),
%  	 0)).

% ** Cannibals-Missionaries **
 get_example(state((3,3,1),0)).

