:-module(pqueue,
	[
	 %% predicates
	 pq_empty/1, pq_enqueue/3, pq_enqueue_union/3,
	 pq_enqueue_all/3,pq_enqueue_all_union/3, pq_dequeue/3,pq_member/2,
	 pq_delete/3,
	 %% regtypes
	 pqueue/1
	],
	[assertions,isomodes,regtypes]).

:-use_module(library(lists)).

:- comment(title, "Priority Queue").
:- comment(author,"Jorge Navas").
:- comment(module,"This module provides a set of predicates for
	           @index{priority queue}s processing. It uses the library
	           @lib{lists} defined in the @concept{Ciao-Prolog System}.").

%% Predicates


:- pred pq_empty(?PQ):: pqueue 
#" @var{PQ} is empty. 

   @item @em{Code:}
   @includedef{pq_empty/1}
". 

pq_empty([]).

:- pred pq_enqueue(-E,-PQ1,+PQ2):: term * pqueue * pqueue
#"Adds the element @var{E} to @var{PQ1} returning @var{PQ2}.

 @item @em{Code:}
 @includedef{pq_enqueue/3}.
".

pq_enqueue(E,[],[E]).
pq_enqueue(E,[H|T],[E,H|T]):- compare(<,E,H),!.
pq_enqueue(E,[H|T],[H|R]):- pq_enqueue(E,T,R).

:- pred pq_enqueue_union(-E,-PQ1,+PQ2):: term * pqueue * pqueue
#"Adds the element @var{E} to @var{PQ1} returning @var{PQ2}. When @var{E}
 belongs to @var{PQ1}, @var{E} is not added to @var{PQ2}.

 @item @em{Code:}
 @includedef{pq_enqueue_union/3}.
".

pq_enqueue_union(E,[],[E]).
pq_enqueue_union(E,[E|T],[E|T]):- !.
pq_enqueue_union(E,[H|T],[E,H|T]):- compare(<,E,H),!.
pq_enqueue_union(E,[H|T],[H|R]):- pq_enqueue_union(E,T,R).

:- pred pq_enqueue_all(-Es,-PQ1,+PQ2):: list(term) * pqueue * pqueue
#"Adds a list of elements @var{Es} to @var{PQ1} returning @var{PQ2}.

 @item @em{Code:}
 @includedef{pq_enqueue_all/3}.
".

pq_enqueue_all([],PQ,PQ).
pq_enqueue_all([E|Es],PQ1,PQ) :- pq_enqueue(E,PQ1,PQ2), pq_enqueue_all(Es,PQ2,PQ).

:- pred pq_enqueue_all_union(-Es,-PQ1,+PQ2):: list(term) * pqueue * pqueue
#"Adds a list of elements @var{Es} to @var{PQ1} returning @var{PQ2}. When a
 element of @var{Es} belongs to @var{PQ1}, this element is not added to @var{PQ2}.

 @item @em{Code:}
 @includedef{pq_enqueue_all_union/3}.
".

pq_enqueue_all_union([],PQ,PQ).
pq_enqueue_all_union([E|Es],PQ1,PQ) :- pq_enqueue_union(E,PQ1,PQ2), pq_enqueue_all_union(Es,PQ2,PQ).

:- pred pq_dequeue(+E,-PQ1,+PQ2):: term * pqueue * pqueue
#"Returns the first element @var{E} of @var{PQ1}, producing @var{PQ2}
 without @var{E}.

 @item @em{Code:}
 @includedef{pq_dequeue/3}.
".

pq_dequeue(E, [E|PQ],PQ).

:- pred pq_member(?E,-PQ):: term * pqueue 
#"Checks the element @var{E} belongs to @var{PQ}.

 @item @em{Code:}
 @includedef{pq_member/2}.
".

pq_member(E, Q):- member(E,Q).

:- pred pq_delete(-E,-PQ1,+PQ2):: term * pqueue * pqueue
# "@var{PQ2} is @var{PQ1} without the element @var{E}.

 @item @em{Code:}
 @includedef{pq_delete/3}.
".

pq_delete(E, PQ1,PQ2):-  delete(PQ1,E,PQ2).

%% Regtypes

:- regtype pqueue(PQ) #"@var{PQ} is a @concept{priority queue}.".

pqueue(PQ):- list(PQ).
