:-module(n_puzzle,
	[
	 get_children/2,   
	 is_goal/1,   
	 select_node/3,   
	 get_function/2,   
	 print_path/1,
	 board/1,
	 tile/1,
	 position/1
	],
	[functions,assertions,isomodes,regtypes]).

:- include('../operators.pl').
:- use_module(library(hiordlib)).
:- use_module(library(write)).
:- use_module(library(lists)).
:- use_module(library(sort)).
:- use_module('../adt/pqueue.pl').
:- use_module('../adt/set.pl').
:- use_module('../adt/bag.pl').
:- use_module('../adt/queue.pl').

:- comment(title, "8-puzzle game").
:- comment(author,"Mario Mendez").
:- comment(module,"



This module implements the @concept{8-puzzle game}, although it can be
configured to implement a @index{N-puzzle game}.  The @concept{8-puzzle
game} is a simple (one-person) game where tiles numbered 1 through 8 are
moved on a 3-by-3 grid. Any tile adjacent to the blank position can be
moved into the blank position. By moving tiles in sequence we attempt to
reach the goal configuration.
").


:- pred get_children(-Node,+Children):: board * list(board)
#"Returns all valid descendent nodes @var{Children} for a given
@var{Node}.

   @item @em{Code:}
   @includedef{get_children/2}
   @includedef{add_new_child/4}
".

get_children(State):=
	Children1++Children2++Children3++Children4 :-
	add_new_child(-1,0,State,Children1),
	add_new_child(0,-1,State,Children2),
	add_new_child(0, 1,State,Children3),
	add_new_child(1 ,0,State,Children4).

add_new_child(Delta_X,Delta_Y,State,[New_Child]):-
	State = board(_,Board,Empty_Tile,G),
	Empty_Tile = tile(pos(X,Y),empty),
	New_Empty_Position = pos(X+Delta_X,Y+Delta_Y), 
	% Is a valid swap? 
        position(New_Empty_Position),
        Board = Before++[tile(New_Empty_Position,Id)]++After,

	New_Board = Before++[tile(pos(X,Y),Id)]++After,
	New_F = G + 1 + ~heuristic(New_Board),
	New_Empty = tile(New_Empty_Position,empty),
	New_Child = board(New_F,New_Board,New_Empty,G+1),!.

add_new_child(_Delta_X,_Delta_Y,_State,[]).


:- pred get_function(-Board,+Value):: board * num
#"Given a @var{Board} returns @var{Value}=f(x), previously calculated by the
 heuristic.

    @item @em{Code:}
    @includedef{get_function/2}
    @includedef{heuristic/2}
    @includedef{heuristic_/3}
    @includedef{goal/1}
 ".
get_function(board(HF,_,_,_)):= HF.


heuristic(Tiles):= ~foldl(Tiles,0,heuristic_).

heuristic_( Tile,H,H) :- goal(Tile),!.
heuristic_(_Tile,H,H+1).

goal(tile(pos(1,1),1)).
goal(tile(pos(1,2),2)).
goal(tile(pos(1,3),3)).
goal(tile(pos(2,1),8)).
goal(tile(pos(2,2),empty)).
goal(tile(pos(2,3),4)).
goal(tile(pos(3,1),7)).
goal(tile(pos(3,2),6)).
goal(tile(pos(3,3),5)).



:- pred is_goal(-Board):: board 
#"@var{Board} is a goal configuration.

   @item @em{Code:}
   @includedef{is_goal/1}
".
is_goal(board(F,_,_,F)).



:- pred select_node(-Board1, -Boards, +Board2):: board * list(board) * board 
#"True if a state similar to @var{Board1} is in @var{Boards}. @var{Board1} is
similar to @var{Board2} if @bf{Board1 = board(_,Tiles,Empty,_,_)} and
@bf{Board2 = board(_,Tiles,Empty,_,_)}.

    @item @em{Code:}
    @includedef{select_node/3}
".
select_node(board(_,Ts,E,_),Visited,Similar):-
	Similar = board(_,Ts,E,_), 
	nth(_,Visited,Similar).



:- pred print_path(-Path):: queue 
#"Outputs each element of @var{Path}, which contains all visited
states from the start node to a goal one.

   @item @em{Code:}
   @includedef{print_path/1}
   @includedef{print_board/2}
   @includedef{print_tiles/3}
   @includedef{print_tile/1}
".
	
print_path(Path):- q_empty(Path).
print_path(Path):-
	q_dequeue(E,Path,Es),
	print_board(E,3),nl,
	print_path(Es).

print_board(board(HF,Tiles,Empty_Tile,_),Dim):-
	  sort([Empty_Tile|Tiles],Sorted_Tiles),
	  write('f(x)= '),write(HF),nl,
	  print_tiles(Sorted_Tiles,Dim,Dim),nl.

print_tiles([],_,_).
print_tiles([T|Ts],1,Cols):-
	  print_tile(T), nl,
	  print_tiles(Ts,Cols,Cols).
print_tiles([T|Ts],Dim,Cols):-
	  print_tile(T), write(' '),
	  New_Dim is Dim - 1,
	  print_tiles(Ts,New_Dim,Cols).

print_tile(tile(_,empty)):-
	  write(' ').
print_tile(tile(_,Id)):-
	  write(Id).



:- comment(board(B),"
  @begin{itemize}
  @item @em{Description:}
  
  @var{B} is a 4-tuple of the form
  @bf{board(HF,Ts,E,G)}, where @var{HF} is the sum of two components: g(n)
  (which is the value of @var{G}) and h(n), which is a heuristic estimate 
  of the distance from state n to a goal, @var{Ts} contains all tiles 
  positions, except for the empty tile, stored in @var{E} for efficiency.
  @var{G} measures the distance from the actual to the start state.

  @item @em{Code:}
  @includedef{board/1}
  @end{itemize}
").

:- regtype board(B) #"@var{B} is a board".

board(board(HF,Ts,E,G)):-
	nnegint(HF),
	list(Ts,tile),
	tile(E),
	nnegint(G).


:- comment(tile(T),"

  @begin{itemize}
  @item @em{Description:}

  @var{T} is a pair of the form @bf{tile(P,I)} where
  @var{P} is a position in the board and @var{I} is the tile identifier.

  @item @em{Code:}
  @includedef{tile/1}
  @end{itemize}

").

:- regtype tile(T) #"".

tile(T) :- 
	T = tile(P,Id),
	num(Id),
	position(P).


:- comment(position(P),"

  @begin{itemize}
  @item @em{Description:}

  @var{P} is a pair @bf{pos(X,Y)} where @var{X} and @var{Y} are coordinates.
 
  @item @em{Code:}
  @includedef{position/1}
  @end{itemize}

").

:- regtype position(P) #"".

position(P):-
	P = pos(X,Y),
	nnegint(X),
	nnegint(Y),
	X =< 3,
	Y =< 3.

