:-module(cannibals,
	[
	 get_children/2,
	 get_function/2,
	 is_goal/1,
	 select_node/3,	 
	 print_path/1,
	 state/2
	],
	[functions,assertions,isomodes,regtypes]).

:- include('../operators.pl').
:- use_module(library(write)).
:- use_module(library(hiordlib)).
:- use_module(library(lists)).
:- use_module('../adt/queue.pl').
:- use_module('../utils.pl').


:- comment(title, "Missionaries & Cannibals problem").
:- comment(module,"

@section{Background}
Implementation the @concept{missionaries/cannibals game}, where 3 missionaries,
 3 cannibals and one boat are in the left bank of the river and should be moved
 using the 2-people capacity boat to the right side. A state where there are
 more cannibals than missionaries is not permitted, whether it is in the right 
or left bank.
").


:- comment(state((M,C,B),G),"

  @begin{itemize}
  @item @em{Description:}

  @begin{itemize}  
  @item @var{M} Number of missionaries
  @item @var{C} Number of cannibals
  @item @var{B} Whether the boat is on the left side of the river or not
  @item @var{G} Distance from the actual to the initial state
  @end{itemize}

  @item @em{Code:}
  @includedef{state/2}
  @includedef{state_/2}
  @includedef{people/1}
  @end{itemize}

").
:- regtype state(S,G) #"@var{S} is a valid configuration for missionaries/cannibals problem with associated g(n)=@var{G}".

state(Tuple,_G) :- state_(Tuple,0).

state_(_,2) :- !.
state_((0,C,_),_Counter):-
	people(C),
	!.
state_((M,C,_),Counter):-
 	people(M),
	people(C),
	M>=C,
	state_((3-M ,3-C,_),Counter+1).

people(N):- 
	nnegint(N),
	N>=0,
	N=<3.


:- pred get_children(-Node,+Children):: board * list(board)
#"Returns all the (valid) descendent nodes @var{Children} of @var{Node}. 
The order in the list has influence in the performance of the dfs algorithm.

@item @em{Code:}
@includedef{get_children/2}
@includedef{add_new_child/3}
".

get_children(State):=
	NewState1++NewState2++NewState3++NewState4++NewState5 :- 
	State = state((_M,_C,1),_G),
	add_new_child((0,-1,0),State,NewState1),
	add_new_child((0,-2,0),State,NewState2),
	add_new_child((-1,-1,0),State,NewState3),
	add_new_child((-1,0,0),State,NewState4),
	add_new_child((-2,0,0),State,NewState5).
	
get_children(State):=
	NewState1++NewState2++NewState3++NewState4++NewState5 :- 
	State = state((_M,_C,0),_G),
	add_new_child((0,1,1),State,NewState1),
	add_new_child((0,2,1),State,NewState2),
	add_new_child((1,1,1),State,NewState3),
	add_new_child((1,0,1),State,NewState4),
	add_new_child((2,0,1),State,NewState5).


add_new_child((Delta_M,Delta_C,New_B),state((M,C,_B),G)):=
	[state((M+Delta_M,C+Delta_C,New_B),G+1)]:-
        state((M+Delta_M,C+Delta_C,New_B),G+1),!.
 
add_new_child((_Delta_M,_Delta_C,_Delta_B),_State):= [].


:- pred is_goal(?Board):: board #"@var{Board} is a goal scenario.

@item @em{Code:}
@includedef{is_goal/1}
".

is_goal(state((0,0,0),_G)).


:- pred get_function(-State,+Value):: state * num
#"Returns f(@var{State})= @var{Value} 
 
@item @em{Code:}
@includedef{get_function/2}
".
get_function(state((M,C,_B),G)) := G + ~max(M,C).


	
:- pred select_node(-State, -VisitedStates,MatchingState):: state * list(state) * state 
#"True if a node similar to @var{State} is in @var{VisitedStates}. @var{State}
 is similar to @var{MatchingState} if both have the same number of missionaries
, cannibals and boats in the left bank

 @item @em{Code:}
 @includedef{select_node/3}
".

select_node(State, Visited_States,state((M,C,B),G)):-
	State = state((M,C,B),_),
	nth(_,Visited_States,state((M,C,B),G)).



:- pred print_path(-Path):: queue 
#"Outputs each element of @var{Path}, which contains all visited
states from the start node to a goal one.

@item @em{Code:}
@includedef{print_path/1}
".

print_path(Path):- q_empty(Path).
print_path(Path):-
	q_dequeue(state((M,C,B),G),Path,Rest),
	write('  [F='),write(~get_function(state((M,C,B),G))),write('] '),
	write(M),write(','),write(C),write(','),write(B),
	nl,
	print_path(Rest).
