:-module(queue,
	[
	 %% predicates
	 q_empty/1, q_enqueue/3, q_enqueue_union/3, q_enqueue_all /3,
	 q_enqueue_all_union /3, q_dequeue/3,q_member/2, q_delete/3,
	 q_length/2,
	 %% regtypes
	 queue/1
	],
	[assertions,isomodes,regtypes]).

:-use_module(library(lists)).

:- comment(title, "Queue").
:- comment(author,"Jorge Navas").
:- comment(module,"This module provides a set of predicates for
	           @index{queue}s processing. It uses the library
	           @lib{lists} defined in the @concept{Ciao-Prolog
	           System}.").

%% Predicates

:- pred q_empty(?Q):: queue
#" @var{Q} is empty. 

   @item @em{Code:}
   @includedef{q_empty/1}
".

q_empty([]).

:- pred q_enqueue(-E,-Q1,+Q2):: term * queue * queue
#"Adds the element @var{E} to @var{Q1} returning @var{Q2}.

   @item @em{Code:}
   @includedef{q_enqueue/3}
".

q_enqueue(E,Q1,Q2):- insert_last(Q1,E,Q2).

:- pred q_enqueue_union(-E,-Q1,+Q2):: term * queue * queue
#"Adds the element @var{E} to @var{Q1} returning @var{Q2}. When @var{E}
 belongs to @var{Q1}, @var{E} is not added to @var{Q2}.

   @item @em{Code:}
   @includedef{q_enqueue_union/3}
".

q_enqueue_union(E,[],[E]).
q_enqueue_union(E,[E|Q],[E|Q]):-!.
q_enqueue_union(E,[Q1|Q1s],[Q1|Q]):- q_enqueue_union(E,Q1s,Q).

:- pred q_enqueue_all(-Es,-Q1,+Q2):: list(term) * queue * queue
#"Adds a list of elements @var{Es} to @var{Q1} returning @var{Q2}.

   @item @em{Code:}
   @includedef{q_enqueue_all/3}
".

q_enqueue_all([],Q,Q).
q_enqueue_all([E|Es],Q1,Q):- insert_last(Q1,E,Q2), q_enqueue_all(Es,Q2,Q).

:- pred q_enqueue_all_union(-Es,-Q1,+Q2):: list(term) * queue * queue
#"Adds a list of elements @var{Es} to @var{Q1} returning @var{Q2}. When a
 element of @var{Es} belongs to @var{Q1}, this element is not added to @var{Q2}.

   @item @em{Code:}
   @includedef{q_enqueue_all_union/3}
".

q_enqueue_all_union([],Q,Q).
q_enqueue_all_union([E|Es],Q1,Q):- q_enqueue_union(Q1,E,Q2), q_enqueue_all_union(Es,Q2,Q).

:- pred q_dequeue(+E,-S1,+S2):: term * queue * queue

#"Returns the first element @var{E} of @var{S1}, producing @var{S2} without
 @var{E}.

   @item @em{Code:}
   @includedef{q_dequeue/3}
".

q_dequeue(E, [E|Q],Q).

:- pred q_member(?E,-Q):: term * queue 
#"Checks the element @var{E} belongs to @var{Q}.

   @item @em{Code:}
   @includedef{q_member/2}
".

q_member(E,Q):- member(E,Q).

:- pred q_delete(-E,-Q1,+Q2):: term * queue * queue
# "@var{Q2} is @var{Q1} without the element @var{E}.

   @item @em{Code:}
   @includedef{q_delete/3}
".

q_delete(E,Q1,Q2):-  delete(Q1,E,Q2).


:- pred q_length(L,N) : list * var => list * integer
# "Computes the length of @var{L}.

   @item @em{Code:}
   @includedef{q_length/2}

".

q_length(L,N):- length(L,N).

%% Regtypes

:- regtype queue(Q) #"@var{Q} is a @concept{queue}.".

queue(Q):- list(Q).
