:-module(bag,
	[
	 %% predicates
	 empty/2, insert/4, insert_union/4, insert_all/4,
	 insert_all_union/4, head/4,member/3, delete/4,
	 %% regtypes
	 bag/1,
	 type_bag/1
	],
	[assertions,isomodes,regtypes]).

:-use_module('./stack').
:-use_module('./queue').
:-use_module('./pqueue').

:- comment(title, "Bag").
:- comment(author,"Jorge Navas").
:- comment(module,"This module provides a set of predicates for
	           @index{bag}s processing. It uses the libraries
	           @lib{stack}, @lib{queue} and @lib{pqueue} defined in
	           this manual").

%% Predicates

:- pred empty(?Bag,-Type):: bag * type_bag
#"@var{Bag} is empty. @var{Bag} is of type @var{Type}.

  @item @em{Code:}
  @includedef{empty/2}
".
 
empty(B,stack) :- s_empty(B).
empty(B,queue):-  q_empty(B).
empty(B,pqueue):- pq_empty(B).


:- pred insert(-E,-B1,+B2,-Type):: term * bag * bag * type_bag
#"Inserts @var{E} into @var{B1} producing @var{B2}, where both @var{B1} and
 @var{B2} are of type @var{Type}.

 @item @em{Code:}
 @includedef{insert/4}.

".

insert(E,B1,B2,stack) :- s_stack(E,B1,B2).
insert(E,B1,B2,queue) :- q_enqueue(E,B1,B2).
insert(E,B1,B2,pqueue) :-  pq_enqueue(E,B1,B2).

:- pred insert_union(-E,-B1,+B2,-Type):: term * bag * bag * type_bag
#"Inserts @var{E} into @var{B1} producing @var{B2}, where both @var{B1} and
 @var{B2} are of type @var{Type}. When @var{E} belongs to @var{B1}, @var{E}
 is not added to @var{B2}.

 @item @em{Code:}
 @includedef{insert_union/4}.

".

insert_union(E,B1,B2,stack) :- s_stack_union(E,B1,B2).
insert_union(E,B1,B2,queue) :- q_enqueue_union(E,B1,B2).
insert_union(E,B1,B2,pqueue) :-  pq_enqueue_union(E,B1,B2).

:- pred insert_all(-Es,-B1,+B2,-Type):: list(term) * bag * bag * type_bag
#"Inserts a list of elements @var{Es} into @var{B1} producing @var{B2}, where both @var{B1} and
 @var{B2} are of type @var{Type}.

 @item @em{Code:}
 @includedef{insert_all/4}.

".

insert_all(Es,B1,B2,stack) :- s_stack_all(Es,B1,B2).
insert_all(Es,B1,B2,queue) :- q_enqueue_all(Es,B1,B2).
insert_all(Es,B1,B2,pqueue) :-  pq_enqueue_all(Es,B1,B2).

:- pred insert_all_union(-Es,-B1,+B2,-Type):: list(term) * bag * bag * type_bag
#"Inserts a list of elements @var{Es} into @var{B1} producing @var{B2},
 where both @var{B1} and @var{B2} are of type @var{Type}. When a element of
 @var{Es} belongs to @var{B1}, this element is not added to @var{B2}.

 @item @em{Code:}
 @includedef{insert_all_union/4}.
".

insert_all_union(Es,B1,B2,stack) :- s_stack_all_union(Es,B1,B2).
insert_all_union(Es,B1,B2,queue) :- q_enqueue_all_union(Es,B1,B2).
insert_all_union(Es,B1,B2,pqueue) :-  pq_enqueue_all_union(Es,B1,B2).


:- pred head(+E,-B1,+B2,-Type):: term * bag * bag * type_bag
#"@var{B2} is @var{B1} without its head element @var{E} where both @var{B1}
  and @var{B2} are of type @var{Type}.

 @item @em{Code:}
 @includedef{head/4}.
".

head(E,B1,B2,stack):- s_stack(E,B2,B1).
head(E,B1,B2,queue):- q_dequeue(E,B1,B2).
head(E,B1,B2,pqueue):- pq_dequeue(E,B1,B2).

:- pred member(?E,-B,-Type):: term * bag * type_bag 
#"Checks the element @var{E} belongs to @var{B} of type @var{Type}.

 @item @em{Code:}
 @includedef{member/3}.
".

member(E,S,stack):- s_member(E,S).
member(E,S,queue):- q_member(E,S).
member(E,S,pqueue):- pq_member(E,S).

:- pred delete(-E,-B1,+B2,-Type):: term * bag * bag * type_bag
# "@var{B2} is @var{B1} without the element @var{E}, where both @var{B1}
 and @var{B2} are of type @var{Type}.

 @item @em{Code:}
 @includedef{delete/4}.
".

delete(E, B1,B2,stack):-  s_delete(E,B1,B2).
delete(E, B1,B2,queue):-  q_delete(E,B1,B2).
delete(E, B1,B2,pqueue):- pq_delete(E,B1,B2).


%% Regtypes

:- regtype bag(B) #"@var{B} is a @concept{data structure} that can be:
@concept{stack}, @concept{queue} or @concept{priority queue}.".

bag(B) :- stack(B).
bag(B) :- queue(B).
bag(B) :- pqueue(B).

:- regtype type_bag(T) #"@var{T} is atom that can be: @concept{stack},
@concept{queue} or @concept{pqueue}.".

type_bag(stack).
type_bag(queue). 
type_bag(pqueue). 


